<?php

namespace App\Http\Controllers;

use App\Models\VehicleUsage;
use App\Models\Vehicle;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class VehicleUsageController extends Controller
{
    public function index()
    {
        /*$vehicleUsages = VehicleUsage::with('vehicle')
            ->where('user_id', Auth::id())
            ->latest()
            ->paginate(10);*/
        $vehicleUsages = VehicleUsage::where('user_id', Auth::id())
            ->with('vehicle')
            ->orderBy('date', 'desc')
            ->get();

    // Calculate total kilometers (similar to your timesheet total calculation)
    $totalKilometers = 0;
    $vehicleWiseTotals = [];

    foreach ($vehicleUsages as $usage) {
        $tripDistance = $usage->end_km - $usage->start_km;
        $totalKilometers += $tripDistance;
        
        // Also calculate per vehicle
        if (!isset($vehicleWiseTotals[$usage->vehicle_id])) {
            $vehicleWiseTotals[$usage->vehicle_id] = 0;
        }
        $vehicleWiseTotals[$usage->vehicle_id] += $tripDistance;
    }

    return view('vehicle-usage.index', compact(
        'vehicleUsages', 
        'totalKilometers', 
        'vehicleWiseTotals'
    ));

        return view('vehicle-usage.index', compact('vehicleUsages'));
    }

    public function create()
    {
        $vehicles = Vehicle::active()->get();
        return view('vehicle-usage.create', compact('vehicles'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'date' => 'required|date',
            'start_km' => 'required|numeric|min:0',
            'end_km' => 'required|numeric|gt:start_km',
            'fuel_consumption' => 'required|numeric|min:0',
            'trip_purpose' => 'required|string|max:500',
            'receipt' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        // Handle receipt upload
        $receiptPath = null;
        if ($request->hasFile('receipt')) {
            $receiptPath = $request->file('receipt')->store('receipts', 'public');
        }

        $vehicleUsage = VehicleUsage::create([
            'vehicle_id' => $request->vehicle_id,
            'date' => $request->date,
            'start_km' => $request->start_km,
            'end_km' => $request->end_km,
            'fuel_consumption' => $request->fuel_consumption,
            'trip_purpose' => $request->trip_purpose,
            'receipt_path' => $receiptPath,
            'user_id' => Auth::id(),
        ]);

        // Update vehicle's current kilometer
        $vehicle = Vehicle::find($request->vehicle_id);
        $vehicle->current_km = $request->end_km;
        $vehicle->save();

        return redirect()->route('vehicle-usage.index')
            ->with('success', 'Vehicle usage record created successfully!');
    }

    public function show($id)
    {
        $vehicleUsage = VehicleUsage::with(['vehicle', 'user'])->findOrFail($id);

        return view('vehicle-usage.show', compact('vehicleUsage'));
    }

    public function edit(VehicleUsage $vehicleUsage)
    {
        // Check if user owns this record
        if ($vehicleUsage->user_id !== Auth::id()) {
            abort(403);
        }

        $vehicles = Vehicle::active()->get();
        return view('vehicle-usage.edit', compact('vehicleUsage', 'vehicles'));
    }

    public function update(Request $request, VehicleUsage $vehicleUsage)
    {
        // Check if user owns this record
        if ($vehicleUsage->user_id !== Auth::id()) {
            abort(403);
        }

        $request->validate([
            'vehicle_id' => 'required|exists:vehicles,id',
            'date' => 'required|date',
            'start_km' => 'required|numeric|min:0',
            'end_km' => 'required|numeric|gt:start_km',
            'fuel_consumption' => 'required|numeric|min:0',
            'trip_purpose' => 'required|string|max:500',
            'receipt' => 'nullable|file|mimes:jpg,jpeg,png,pdf|max:2048',
        ]);

        // Handle receipt upload
        $receiptPath = $vehicleUsage->receipt_path;
        if ($request->hasFile('receipt')) {
            // Delete old receipt if exists
            if ($receiptPath) {
                Storage::disk('public')->delete($receiptPath);
            }
            $receiptPath = $request->file('receipt')->store('receipts', 'public');
        }

        $vehicleUsage->update([
            'vehicle_id' => $request->vehicle_id,
            'date' => $request->date,
            'start_km' => $request->start_km,
            'end_km' => $request->end_km,
            'fuel_consumption' => $request->fuel_consumption,
            'trip_purpose' => $request->trip_purpose,
            'receipt_path' => $receiptPath,
        ]);

        // Update vehicle's current kilometer
        $vehicle = Vehicle::find($request->vehicle_id);
        $vehicle->current_km = $request->end_km;
        $vehicle->save();

        return redirect()->route('vehicle-usage.index')
            ->with('success', 'Vehicle usage record updated successfully!');
    }

    public function destroy(VehicleUsage $vehicleUsage)
    {
        // Check if user owns this record
        if ($vehicleUsage->user_id !== Auth::id()) {
            abort(403);
        }

        // Delete receipt file if exists
        if ($vehicleUsage->receipt_path) {
            Storage::disk('public')->delete($vehicleUsage->receipt_path);
        }

        $vehicleUsage->delete();

        return redirect()->route('vehicle-usage.index')
            ->with('success', 'Vehicle usage record deleted successfully!');
    }
}