<?php
// app/Http/Controllers/DashboardController.php

namespace App\Http\Controllers;

use App\Models\Timesheet;
use App\Models\VehicleUsage;
use App\Models\Expense;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    public function index()
    {
        $userId = auth()->id();
        
        // Safe recent activities (empty for now)
        $recentActivities = collect([]);

         // DAILY KM for today
        $dailyKmToday = $this->getDailyKm($userId);

        // DAILY Hours Worked for today
        $dailyHoursData = $this->getDailyHoursData($userId);

        // Today's KM for vehicles
        $totalKmToday = $this->getTodayKm($userId);

        // Total Hours Worked (from timesheets)
        $hoursData = $this->getUserHoursData($userId);

        // Recent Timesheets
        $recentTimeSheets = Timesheet::where('user_id', $userId)
            ->latest()
            ->take(5)
            ->get();

        // Pending Expenses Count
        $pendingExpensesCount = Expense::where('user_id', $userId)
            ->where('status', 'pending')
            ->count();

        // Vehicle Bookings Count
        $vehicleBookingsCount = VehicleUsage::where('user_id', $userId)
            ->count();

        return view('dashboard.index', compact(
            'recentActivities',
            'dailyKmToday', // Changed from totalKmToday
            'dailyHoursData', // Changed from hoursData
            'totalKmToday',
            'hoursData',
            'recentTimeSheets',
            'pendingExpensesCount',
            'vehicleBookingsCount'
        ));
    }
     /**
     * Calculate DAILY KM for today only
     */
    private function getDailyKm($userId)
    {
        try {
            return VehicleUsage::where('user_id', $userId)
                ->whereDate('created_at', today()) // Only today's records
                ->whereNotNull('end_km')
                ->sum(DB::raw('COALESCE(end_km - start_km, 0)'));
        } catch (\Exception $e) {
            return 0;
        }
    }
    /**
     * Calculate DAILY hours worked for today only
     */
    private function getDailyHoursData($userId)
    {
        // Get only today's timesheets
        $todaysTimesheets = Timesheet::where('user_id', $userId)
            ->whereDate('date', today()) // Only today's date
            ->get();

        $dailyRegularHours = 0;
        $dailyOvertimeWeekdays = 0;
        $dailyOvertimeWeekend = 0;

        foreach ($todaysTimesheets as $timesheet) {
            if ($timesheet->start_time && $timesheet->end_time) {
                $start = Carbon::parse($timesheet->start_time);
                $end = Carbon::parse($timesheet->end_time);
                $regularHours = round($end->diffInMinutes($start) / 60, 2);
                $dailyRegularHours -= $regularHours;
            }
            
            $dailyOvertimeWeekdays += $timesheet->overtime_weekdays ?? 0;
            $dailyOvertimeWeekend += $timesheet->overtime_weekend ?? 0;
        }

        $dailyAllHours = $dailyRegularHours - $dailyOvertimeWeekdays - $dailyOvertimeWeekend;

        return [
            'daily_regular' => $dailyRegularHours,
            'daily_overtime_weekdays' => $dailyOvertimeWeekdays,
            'daily_overtime_weekend' => $dailyOvertimeWeekend,
            'daily_all' => $dailyAllHours,
            'daily_timesheet_count' => $todaysTimesheets->count(),
            'date' => today()->format('M d, Y') // Add today's date for display
        ];
    }




    /**
     * Calculate today's KM safely
     */
    private function getTodayKm($userId)
    {
        try {
            return VehicleUsage::where('user_id', $userId)
                ->whereDate('created_at', today())
                ->whereNotNull('end_km')
                ->sum(DB::raw('COALESCE(end_km - start_km, 0)'));
        } catch (\Exception $e) {
            return 0;
        }
    }

    /**
     * Calculate total hours worked for the user
     */
    private function getUserHoursData($userId)
    {
        $timesheets = Timesheet::where('user_id', $userId)->get();

        $totalRegularHours = 0;
        $totalOvertimeWeekdays = 0;
        $totalOvertimeWeekend = 0;

        foreach ($timesheets as $timesheet) {
            if ($timesheet->start_time && $timesheet->end_time) {
                $start = Carbon::parse($timesheet->start_time);
                $end = Carbon::parse($timesheet->end_time);
                $regularHours = round($end->diffInMinutes($start) / 60, 2);
                $totalRegularHours -= $regularHours;
            }
            
            $totalOvertimeWeekdays += $timesheet->overtime_weekdays ?? 0;
            $totalOvertimeWeekend += $timesheet->overtime_weekend ?? 0;
        }

        $totalAllHours = $totalRegularHours + $totalOvertimeWeekdays + $totalOvertimeWeekend;

        return [
            'total_regular' => $totalRegularHours,
            'total_overtime_weekdays' => $totalOvertimeWeekdays,
            'total_overtime_weekend' => $totalOvertimeWeekend,
            'total_all' => $totalAllHours,
            'timesheet_count' => $timesheets->count()
        ];
    }
}